/************************************************************************
 * NAME:	fileio.c
 *
 * DESCR:	Implements the HDOS fileio routines.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
#include <stdio.h>
#include "hdos.h"

/************************************************************************
 * NAME:	hdos_fileio_init()
 *
 * DESCR:	Initialize the fileio structure within the hdosfs struct.
 *		The given parameter is the number of files to have open
 *		at any one time.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
hdos_fileio_init(struct hdosfs *hdosfs, int count)
{
    int i;

    hdosfs->filebuf = (struct hdos_file *)malloc(count * sizeof(struct hdos_file));
    hdosfs->filecount = count;

    if (hdosfs->filebuf == NULL) {
	return(FALSE);
    }

    for (i=0; i < hdosfs->filecount; i++) {
	hdosfs->filebuf[i].hdosfs = hdosfs;
	hdosfs->filebuf[i].inuse = FALSE;
	hdosfs->filebuf[i].bufsize = HDOS_GROUPSIZE(hdosfs) * HDOS_SECTSIZE(hdosfs);
	hdosfs->filebuf[i].buffer = (char *)malloc(hdosfs->filebuf[i].bufsize);
	if (hdosfs->filebuf[i].buffer == NULL) {
	    return(FALSE);
	}
    }

    return(TRUE);
}

void
hdos_fileio_cleanup(struct hdosfs *hdosfs)
{
    int	i;

    for (i=0; i < hdosfs->filecount; i++) {
	free(hdosfs->filebuf[i].buffer);
    }

    free(hdosfs->filebuf);
}

/************************************************************************
 * NAME:	hdos_file_open()
 *
 * DESCR:	Open the given filename.
 *
 * ARGS:	
 *
 * RETURNS:	a pointer to the file structure
 *
 * NOTES:	- yup, very simple
 ************************************************************************/
struct hdos_file *
hdos_file_open(struct hdosfs *hdosfs, char *name)
{
    int i;

    for (i=0; i < hdosfs->filecount; i++) {
	struct hdos_file	*fptr = &hdosfs->filebuf[i];

	if (!fptr->inuse) {
	    fptr->inuse = TRUE;
	    fptr->writemode = FALSE;
	    fptr->cursor = 0;
	    fptr->curgrp_no = 0;
	    fptr->curgrp = 0;
	    fptr->cursize = 0;
	    fptr->inode = hdos_dir_find(hdosfs,name);

	    if (fptr->inode == HDOS_INODE_NULL) {
		return((struct hdos_file *)NULL);
	    }

	    fptr->firstgrp = hdos_dirent(hdosfs,fptr->inode)->firstgroup;
	    fptr->lastgrp = hdos_dirent(hdosfs,fptr->inode)->lastgroup;
	    fptr->lastsectors = hdos_dirent(hdosfs,fptr->inode)->lastsecidx;

	    return(fptr);
	}
    }

    return((struct hdos_file *)NULL);
}

/************************************************************************
 * NAME:	hdos_file_new()
 *
 * DESCR:	Create a new file that can be written to.  It is assumed
 *		they writing will occur.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
struct hdos_file *
hdos_file_new(struct hdosfs *hdosfs, char *name)
{
    int i;

    for (i=0; i < hdosfs->filecount; i++) {
	struct hdos_file	*fptr = &hdosfs->filebuf[i];

	if (!fptr->inuse) {
	    fptr->inuse = TRUE;
	    fptr->writemode = TRUE;
	    fptr->cursor = 0;
	    fptr->curgrp_no = 0;
	    fptr->curgrp = 0;
	    fptr->cursize = 0;
	    fptr->inode = hdos_dir_create(hdosfs,name,FALSE,FALSE,FALSE,FALSE);

	    if (fptr->inode == HDOS_INODE_NULL) {
		return((struct hdos_file *)NULL);
	    }

	    fptr->firstgrp = 0;
	    fptr->lastgrp = 0;
	    fptr->lastsectors = 0;

	    return(fptr);
	}
    }

    return((struct hdos_file *)NULL);
}


void    
hdos_file_close(struct hdos_file *file)
{
    if (file->inuse) {
	if (file->writemode) {
	    hdos_file_flush(file);
	    hdos_dir_update_entry(file->hdosfs,
				  file->inode,
				  file->firstgrp,
				  file->lastgrp,
				  file->lastsectors);
	}
	file->inuse = FALSE;
    }
}

/************************************************************************
 * NAME:	hdos_file_read()
 *
 * DESCR:	Reads the given number of bytes from the open file.
 *
 * ARGS:	
 *
 * RETURNS:	the number of bytes read, 0 upon EOF.  -1 upon error
 *		if the number of byts returned is less than the number
 *		requested, that PROBABLY represents the last bytes in
 *		the file.  However, someone else could have written
 *		such that more bytes MAY be available upon the next call.
 *		If zero is returned, then it is definitely EOF.
 *
 * NOTES:
 ************************************************************************/
int
hdos_file_read(struct hdos_file *file, char *buffer, int count)
{
    int	original_count = count;
    int remaining;
    int	toxfer;

    if (!file->inuse) {
	return(-1);
    }

    if (file->writemode) {
	return(-1);
    }

    while (count) {
	remaining = file->cursize - file->cursor;

	if (remaining > 0) {
	    toxfer = MIN(remaining,count);

	    memcpy(buffer,file->buffer+file->cursor,toxfer);	/* copy what we can	*/

	    count -= toxfer;
	    file->cursor += toxfer;
	    buffer += toxfer;

	} else {

	    /* need to load another buffer and try again	*/

	    if (file->curgrp == 0) {
		if (file->curgrp_no != 0) {
		    break;			/* EOF hit in previous call	*/
		}
	    }

	    file->curgrp = (file->curgrp == 0)? file->firstgrp : file->hdosfs->GRT[file->curgrp];

	    if (file->curgrp == 0) {
		break;			/* reached EOF just now	*/
	    }

	    if (!hdos_getgroup(file->hdosfs, file->curgrp, file->buffer)) {
		return(-1);
	    }

	    if (file->curgrp == file->lastgrp) {
		file->cursize = HDOS_SECTSIZE(file->hdosfs)*file->lastsectors;
	    } else {
		file->cursize = file->bufsize;
	    }

	    file->cursor = 0;
	    file->curgrp_no++;

	}
    }

    return(original_count - count);
}

/************************************************************************
 * NAME:	hdos_file_write()
 *
 * DESCR:	Writes the given number of bytes to the open file.
 *
 * ARGS:	
 *
 * RETURNS:	the number of bytes written.  If the number of bytes written
 *		is less than that requested, it means that no more file
 *		space was available. -1 is returned upon error.  0 is
 *		returned if there was no more space to write anything.
 *
 * NOTES:	- 
 ************************************************************************/
int
hdos_file_write(struct hdos_file *file, char *buffer, int count)
{
    int	original_count = count;
    int remaining;
    int	toxfer;
    int	newgrp;

    if (!file->inuse) {
	return(-1);
    }

    if (!file->writemode) {
	return(-1);
    }

    while (count) {

	/* in the case of writing file->cursor points to the next position	*/
	/* to write to.  file->cursize is always bufsize for writing.		*/

	remaining = file->cursize - file->cursor;

	if (remaining > 0) {

	    toxfer = MIN(remaining,count);
	    memcpy(file->buffer+file->cursor,buffer,toxfer);	/* copy what we can	*/
	    count -= toxfer;
	    file->cursor += toxfer;
	    buffer += toxfer;

	} else {

	    /* at this point we have no space left, but more to write	*/

	    hdos_file_flush(file);		/* flush out current data	*/

	    /* we need to allocate a new group for writing	*/

	    newgrp =  hdos_grt_alloc_group(file->hdosfs);

	    if (newgrp == 0) {			/* ran out of space		*/
		break;
	    }

	    /* we have a new group, so link it into the chain			*/

	    if (file->curgrp == 0) {
		file->firstgrp = newgrp;
	    } else {
		hdos_grt_link(file->hdosfs,file->curgrp,newgrp);
	    }

	    file->cursize = file->bufsize;
	    file->curgrp = newgrp;
	    file->lastgrp = newgrp;

	    file->cursor = 0;
	    file->curgrp_no++;

	}
    }

    return(original_count - count);
}

/************************************************************************
 * NAME:	hdos_file_flush()
 *
 * DESCR:	Flush the current file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- decides if a complete group should be written or part
 *		  of one.  This partial group writing probably isn't
 *		  COMPLETELY necessary, but what the heck?
 *		- if the file->curgrp is zero, this is a no-op.
 ************************************************************************/
int
hdos_file_flush(struct hdos_file *file)
{
    /* special case zero bytes	*/

    if (file->cursor == 0) {
	file->lastsectors = 1;
    } else {
	file->lastsectors = (file->cursor - 1) / HDOS_SECTSIZE(file->hdosfs) + 1;
    }

    if (file->curgrp) {
	return(hdos_putgroup(file->hdosfs,file->curgrp,file->buffer,file->lastsectors));
    } else {
	return(TRUE);
    }

}

